/*
 * paths.c -- these routines compute the path to be traveled by a
 * ball for each type of toss in the pattern.
 */

#include "juggle.h"
#include "proto.h"
#include "paths.h"

#ifdef SHOWPATH
#include <graphics.h>
#endif

void calccatchpos(toss *pattern, int len)
{
   int i, j;

   for (i=0; i < len; i++)
      switch (pattern[i].height) {
         case 0:
         case 1:
            j = (i + pattern[i].height) % len;
            pattern[i].to = pattern[i].nextfrom = pattern[j].from;
            break;
         case 2:
            if (hold_two_flag) {
               j = (i + 2) % len;
               pattern[i].to = pattern[i].nextfrom = pattern[j].from;
               break;
            }                                      /* else fall through */
         default:
            j = (i + pattern[i].height - 2) % len;
            pattern[i].to = !pattern[j].from;
            pattern[i].nextfrom = pattern[(j+2) % len].from;
            break;
      }
}


int calc_count(int height)
{
   switch (height) {
      case 0:
      case 1:
         return height * height_counts;
      case 2:
         if (hold_two_flag)
            return 2 * height_counts;             /* else fall through */
      default:
         return height * height_counts - catch_counts;
   }
}

float calcpaths(toss pattern[], int len)
{
   int i, j;
   windowpt start, end;
   float max, g;

   for (i=0, max=(float)calc_count(3); i < len; i++)
      if ((pattern[i].count = calc_count(pattern[i].height)) > max)
         max = (float)pattern[i].count;
   max /= height_counts;
   windowsize(max*max + 1, -max/2.0 - 1, max+1);

   for (i=0; i < len; i++) {

      for (j=0; j < i; j++)                     /* Check if the path    */
         if (sametoss(pattern+i, pattern+j))    /* was computed earlier */
            break;                              
      if (j < i) {                              /* (j < i) only if the  */
         pattern[i].path = pattern[j].path;     /* loop broke early, ie */
         pattern[i].tofree = NO;                /* a match was found    */
         continue;                              
      }

      pattern[i].path = (screenpt *)smalloc(pattern[i].count * sizeof(screenpt));
      pattern[i].tofree = YES;
      start = hand[pattern[i].from];
      end = hand[pattern[i].to];
      if (pattern[i].height % 2)
         end.x = -end.x;
      g = (pattern[i].height == 2 && hold_two_flag) ? 0 : GRAVITY;

      calcparabola(start, end, pattern[i].count, g, pattern[i].path);

   }
   return max;
}

int sametoss(toss *a, toss *b)
{
   return ( (a->height == b->height) &&
            (a->from   == b->from)   &&
            (a->to     == b->to)     );
}


void calcparabola(windowpt p0, windowpt p1, int count, float g, screenpt *loc)
{
   int i;
   float t, tmax, dx, dy, dt;

   if (!count) return;              /* for count=0, nothing to compute   */

   dt = 1 / (float)height_counts;
   tmax = count * dt;               /* tmax is a little less than height */
   dx = (p1.x - p0.x) / tmax;
   dy = (p1.y - p0.y) / tmax;

   for (i=0, t=0; i < count; i++, t += dt, loc++) {
      loc->x = screenx(p0.x + t*dx);
      loc->y = screeny(p0.y + t*dy - g * (t) * (tmax - t));

#ifdef SHOWPATH
      putpixel(loc->x, loc->y, 15);
      putpixel(mirrorx(loc->x), loc->y, 15);
#endif

   }
}


void calchandpaths(float push)
{
   int i, j;

   for (i=0; i < HANDPOS; i++)
      for (j=0; j < HANDPOS; j++) {
         handpath[i][j] = (screenpt *)smalloc(catch_counts * sizeof(screenpt));
         calcparabola(hand[i], hand[j], catch_counts, push, handpath[i][j]);
      }
}

/*
 * releasepaths releases all memory malloced by calcpaths or calchandpaths.
 */

void releasepaths(toss *pattern, int len)
{
   int i, j;

   for (i=0; i < len; i++)
      if (pattern[i].tofree)
         free (pattern[i].path);

   for (i=0; i < HANDPOS; i++)
      for (j=0; j < HANDPOS; j++)
         free (handpath[i][j]);
}


